import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/date_converter.dart';
import 'package:viserpay/core/helper/string_format_helper.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_icon.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/style.dart';
import 'package:viserpay/data/controller/auto_payment/auto_payment_controller.dart';
import 'package:viserpay/data/model/auto_payment/auto_paymet_response_model.dart';
import 'package:viserpay/data/repo/auto_paymnet/auto_payment_repo.dart';
import 'package:viserpay/data/services/api_service.dart';
import 'package:viserpay/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay/view/components/bottom-sheet/custom_bottom_sheet.dart';
import 'package:viserpay/view/components/custom_loader/custom_loader.dart';
import 'package:viserpay/view/components/image/custom_svg_picture.dart';
import 'package:viserpay/view/components/no_data.dart';
import 'package:viserpay/view/screens/auto_payment/widget/auto_payment_bill_pay_bottom_sheet.dart';
import 'package:viserpay/view/screens/auto_payment/widget/auto_payment_mobile_recharge_bottom_sheet.dart';
import 'package:viserpay/view/screens/auto_payment/widget/auto_payment_send_money_bottom_sheet.dart';
import 'package:viserpay/view/screens/auto_payment/widget/select_auto_payment_category_bottom_sheet.dart';

class AutoPaymentScreen extends StatefulWidget {
  const AutoPaymentScreen({super.key});

  @override
  State<AutoPaymentScreen> createState() => _AutoPaymentScreenState();
}

class _AutoPaymentScreenState extends State<AutoPaymentScreen> with SingleTickerProviderStateMixin {
  late TabController _tabController;

  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(AutoPaymentRepo(apiClient: Get.find()));
    final controller = Get.put(AutoPaymentController(autoPaymentRepo: Get.find()));

    super.initState();
    _tabController = TabController(length: 3, vsync: this);

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.initialData();
    });
  }

  @override
  Widget build(BuildContext context) {
    return GetBuilder<AutoPaymentController>(builder: (controller) {
      return Scaffold(
        backgroundColor: MyColor.screenBgColor,
        appBar: CustomAppBar(
          title: MyStrings.autoPayment,
          isTitleCenter: true,
          elevation: 0.1,
          action: [],
        ),
        body: controller.isLoading
            ? const CustomLoader()
            : DefaultTabController(
                length: 3,
                child: Column(
                  children: [
                    Container(
                      color: MyColor.colorWhite,
                      child: TabBar(
                        controller: _tabController,
                        labelColor: MyColor.primaryColor,
                        dividerColor: MyColor.getContentTextColor().withValues(alpha: 0.1),
                        unselectedLabelColor: MyColor.getContentTextColor(),
                        indicatorColor: MyColor.primaryColor,
                        isScrollable: true,
                        tabAlignment: TabAlignment.center,
                        labelPadding: EdgeInsetsDirectional.symmetric(vertical: Dimensions.space5, horizontal: Dimensions.space10),
                        tabs: [
                          Tab(text: MyStrings.sendMoney.tr),
                          Tab(text: MyStrings.mobileRecharge.tr),
                          Tab(text: MyStrings.payBill.tr),
                        ],
                        onTap: (value) {},
                      ),
                    ),
                    Expanded(
                      child: TabBarView(
                        physics: AlwaysScrollableScrollPhysics(parent: BouncingScrollPhysics()),
                        controller: _tabController,
                        children: [
                          buildListViewForSendMoneyAutoPayment(controller.autoPaymentResponseModelData?.data?.autoPaymentSendMoney ?? [], controller),
                          buildListViewForMobileRechargeAutoPayment(controller.autoPaymentResponseModelData?.data?.autoPaymentMobileRechargeHistoryList ?? [], controller),
                          buildListViewForPayBillAutoPayment(controller.autoPaymentResponseModelData?.data?.autoPaymentPayBillHistoryList ?? [], controller),
                        ],
                      ),
                    ),
                  ],
                ),
              ),
        floatingActionButton: !controller.isLoading
            ? FloatingActionButton.extended(
                onPressed: () {
                  CustomBottomSheet(child: SelectAutoBillPayCategoryBottomSheet()).customBottomSheet(context);
                },
                label: Text(
                  MyStrings.enableNewAutoPay.tr,
                  style: regularDefault.copyWith(color: MyColor.colorWhite),
                ),
                icon: Icon(
                  Icons.add,
                  color: MyColor.colorWhite,
                ),
                backgroundColor: MyColor.getPrimaryColor(),
              )
            : SizedBox.shrink(),
      );
    });
  }

  // Helper method to build a list view for each tab
  Widget buildListViewForSendMoneyAutoPayment(List<AutoPaymentHistoryData> items, AutoPaymentController controller) {
    return RefreshIndicator(
      color: MyColor.getPrimaryColor(),
      onRefresh: () async {
        controller.initialData();
      },
      child: items.isEmpty
          ? NoDataWidget()
          : ListView.builder(
              physics: AlwaysScrollableScrollPhysics(parent: ClampingScrollPhysics()),
              itemCount: items.length,
              padding: const EdgeInsets.all(8.0),
              itemBuilder: (context, index) {
                var item = items[index];
                return Card(
                  margin: const EdgeInsets.symmetric(vertical: 4.0),
                  child: ListTile(
                    onTap: () {
                      CustomBottomSheet(child: AutoPaymentSendMoneyBottomSheet(item: item)).customBottomSheet(context);
                    },
                    title: Text(
                      item.receiver?.getFullName() ?? "",
                      style: boldDefault.copyWith(color: MyColor.getTextColor()),
                    ),
                    subtitle: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Text(
                          "${MyStrings.amount}: ${StringConverter.formatNumber('${item.amount}', precision: 2)}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor().withValues(alpha: 0.8)),
                        ),
                        Text(
                          "${MyStrings.phone}: +${item.receiver?.dialCode ?? ""}${item.receiver?.mobile ?? ""}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor().withValues(alpha: 0.8)),
                        ),
                        Text(
                          "${MyStrings.nextAutoPaymentDate}: ${DateConverter.estimatedDate(item.nextPayment ?? DateTime.now())}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor().withValues(alpha: 0.8)),
                        ),
                      ],
                    ),
                    leading: const CustomSvgPicture(image: MyIcon.sendMoney, color: MyColor.sendMoneyBaseColor, height: 24, width: 24),
                  ),
                );
              },
            ),
    );
  }

  // Helper method to build a list view for each tab
  Widget buildListViewForMobileRechargeAutoPayment(List<AutoPaymentHistoryData> items, AutoPaymentController controller) {
    return RefreshIndicator(
      color: MyColor.getPrimaryColor(),
      onRefresh: () async {
        controller.initialData();
      },
      child: items.isEmpty
          ? NoDataWidget()
          : ListView.builder(
              physics: AlwaysScrollableScrollPhysics(parent: ClampingScrollPhysics()),
              itemCount: items.length,
              padding: const EdgeInsets.all(8.0),
              itemBuilder: (context, index) {
                var item = items[index];
                return Card(
                  margin: const EdgeInsets.symmetric(vertical: 4.0),
                  child: ListTile(
                    onTap: () {
                      CustomBottomSheet(child: AutoPaymentMobileRechargeBottomSheet(item: item)).customBottomSheet(context);
                    },
                    title: Text(
                      "${controller.autoPaymentResponseModelData?.data?.mobileOperators?.firstWhere(
                        (i) {
                          return i.id == item.operatorId;
                        },
                        orElse: () => MobileOperatorCategory(name: ""),
                      ).name}",
                      style: boldDefault.copyWith(color: MyColor.getTextColor()),
                    ),
                    subtitle: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Text(
                          "${MyStrings.amount}: ${Get.find<ApiClient>().getCurrencyOrUsername(isSymbol: true)}${StringConverter.formatNumber('${item.amount}', precision: 2)}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor().withValues(alpha: 0.8)),
                        ),
                        Text(
                          "${MyStrings.phone}: ${item.userData?.mobile ?? ""}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor().withValues(alpha: 0.8)),
                        ),
                        Text(
                          "${MyStrings.nextAutoPaymentDate}: ${DateConverter.estimatedDate(item.nextPayment ?? DateTime.now())}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor().withValues(alpha: 0.8)),
                        ),
                      ],
                    ),
                    leading: const CustomSvgPicture(image: MyIcon.mobileRecharge, color: MyColor.rechargeBaseColor, height: 24, width: 24),
                  ),
                );
              },
            ),
    );
  }

  Widget buildListViewForPayBillAutoPayment(List<AutoPaymentPayBill> items, AutoPaymentController controller) {
    return RefreshIndicator(
      color: MyColor.getPrimaryColor(),
      onRefresh: () async {
        controller.initialData();
      },
      child: items.isEmpty
          ? NoDataWidget()
          : ListView.builder(
              physics: AlwaysScrollableScrollPhysics(parent: ClampingScrollPhysics()),
              itemCount: items.length,
              padding: const EdgeInsets.all(8.0),
              itemBuilder: (context, index) {
                var item = items[index];
                return Card(
                  margin: const EdgeInsets.symmetric(vertical: 4.0),
                  child: ListTile(
                    onTap: () {
                      CustomBottomSheet(child: AutoPaymentBillPayBottomSheet(item: item)).customBottomSheet(context);
                    },
                    title: Text(
                      item.setupUtilityBill?.name ?? "",
                      style: boldDefault.copyWith(color: MyColor.getTextColor()),
                    ),
                    subtitle: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Text(
                          "${MyStrings.amount}: ${Get.find<ApiClient>().getCurrencyOrUsername(isSymbol: true)}${StringConverter.formatNumber('${item.amount}', precision: 2)}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor().withValues(alpha: 0.8)),
                        ),
                        Text(
                          "${MyStrings.nextAutoPaymentDate}: ${DateConverter.estimatedDate(item.nextPayment ?? DateTime.now())}",
                          style: regularDefault.copyWith(color: MyColor.getTextColor().withValues(alpha: 0.8)),
                        ),
                      ],
                    ),
                    leading: const CustomSvgPicture(image: MyIcon.paybill, color: MyColor.paybillBaseColor, height: 24, width: 24),
                  ),
                );
              },
            ),
    );
  }
}
